#include "config.h"
#include <libxml/globals.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxml/HTMLparser.h>
#include <string.h>

void replace_absuri_with_reluri(xmlNodePtr root, xmlChar* base);
xmlChar* convert_absuri_to_reluri(xmlChar *absuri, xmlChar* baseuri);
void traverse_tree(xmlNodePtr children, int indent, xmlChar *base);

typedef enum {
	IFT_XML,
	IFT_HTML
} InputFileType;

int
main(int argc, char* argv[])
{
     const char *params[] = {
	  "path", NULL,
	  NULL
     };
     char* path_param;
     xsltStylesheetPtr style;
     xmlDocPtr doc, res;
     xmlNodePtr root;
     char *stylefile, *inputfile, *baseuri;
     InputFileType filetype = IFT_XML;

     if (argc < 3 || argc > 4) {
	  char *p = strrchr(argv[0], '/');
	  p = p ? p + 1 : argv[0];
	  fprintf(stderr, "usage: %s STYLE XML URI\n"
		          "       %s HTML URI\n", p, p);
	  exit(1);
     }

     if (argc == 4) {
	     stylefile = argv[1];
	     inputfile = argv[2];
	     baseuri = argv[3];
	     filetype = IFT_XML;
     } else {
	     inputfile = argv[1];
	     baseuri = argv[2];
	     filetype = IFT_HTML;
     }

     /* Check URI parameter and initialize parameter for XSLT */
     if (*baseuri != '/') {
	  fprintf(stderr, "Error: URI must be started with '/'.\n");
	  exit(1);
     }
     path_param = xmlMalloc(strlen(baseuri) + 3);
     if (!path_param) {
	  fprintf(stderr, "Error: Failed to allocate memory for parameter\n");
	  exit(1);
     }
     sprintf(path_param, "'%s'", baseuri);
     params[1] = path_param;

     /* Initialize XML parser */
     LIBXML_TEST_VERSION;
     xmlInitMemory();
     xmlSubstituteEntitiesDefault(1);
     exsltRegisterAll();
     xsltRegisterTestModule();
     xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;

     if (filetype == IFT_XML) {
	     /* Parse XML file */
	     doc = xmlParseFile(inputfile);
	     xmlXIncludeProcess(doc);

	     /* Parse style sheet */
	     style = xsltParseStylesheetFile((const xmlChar *)stylefile);
	     res = xsltApplyStylesheet(style, doc, params);
     } else {
	     /* Parse HTML file */
	     res = htmlParseFile(inputfile, NULL);
     }

     /* Rewrite links */
     root = xmlDocGetRootElement(res);
     if (root && root->children) {
	  replace_absuri_with_reluri(root, baseuri);
     }

     if (filetype == IFT_XML) {
	     /* Output results (XML) */
	     xsltSaveResultToFile(stdout, res, style);

	     /* Finalize XSLT/XML */
	     xsltFreeStylesheet(style);
	     xmlFreeDoc(doc);
     } else {
	     /* Output results (HTML) */
	     htmlDocDump(stdout, res);
     }
     xmlFreeDoc(res);
     xsltCleanupGlobals();
     xmlCleanupParser();

     exit(0);
}

void
replace_absuri_with_reluri(xmlNodePtr root, xmlChar* base)
{
     traverse_tree(root->children, 0, base);
}

xmlChar*
convert_absuri_to_reluri(xmlChar *absuri, xmlChar* baseuri)
{
     unsigned char *a, *b, *last_slash, *tmp;
     xmlChar* ret;
     size_t retsize;
     int depth, i;

     a = (unsigned char *)absuri;
     b = (unsigned char *)baseuri;
     if (*absuri != '/' || *baseuri != '/') return NULL;
#if 0
     /* remove /../ and /./ */
     resolve_uri(absuri);
     resolve_uri(baseuri);
#endif
     last_slash = b;
     while (*a == *b && *a && *b) {
	  if (*b == '/') last_slash = b;
	  a++;
	  b++;
     }
     depth = 0;
     tmp = last_slash + 1;
     while (*tmp) {
	  if (*tmp == '/') depth++;
	  tmp++;
     }
     tmp = absuri + (last_slash - baseuri) + 1;
     retsize =  strlen(tmp) + 3 * depth + 1;
     ret = xmlMalloc(retsize);
     if (ret == NULL) return NULL;
     *ret = '\0';
     for (i = 0; i < depth; i++) {
	  strcat(ret, "../");
     }
     strcat(ret, tmp);
     return ret;
}

void
traverse_tree(xmlNodePtr children, int indent, xmlChar *base)
{
     xmlNodePtr node;
     for (node = children; node; node = node->next) {
	  if (node->type == XML_ELEMENT_NODE) {
	       if (strcmp((char *)node->name, "a") == 0 ||
		   strcmp((char *)node->name, "link") == 0) {
		    xmlChar* href = xmlGetProp(node, "href");
		    if (href && href[0] == '/' && href[1] != '/') {
			 xmlChar* newuri;
			 if (newuri = convert_absuri_to_reluri(href, base)) {
			      xmlSetProp(node, "href", newuri);
			      xmlFree(newuri);
			 }
		    }
		    xmlFree(href);
	       }
	       if (node->children)
		    traverse_tree(node->children, indent + 1, base);
	  }
     }
}

/* end of wk */
