/*
 * $Id: GameManager.java,v 1.2 2002/02/08 16:32:42 kenta Exp $
 *
 * Copyright 2002 Kenta Cho. All rights reserved.
 */
package jp.gr.java_conf.abagames.narihira;

import jp.gr.java_conf.abagames.bulletml.*;
import jp.gr.java_conf.abagames.narihira.relaxer.*;
import jp.gr.java_conf.abagames.util.*;
import java.io.*;
import java.util.Vector;
import java.util.Random;
import java.awt.TextArea;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXParseException;
import org.w3c.dom.Document;

/**
 * Handle game status.
 *
 * @version $Revision: 1.2 $
 */
public class GameManager implements ITime {
  public static final double SCREEN_ML_RATE = 0.48;

  private final int BULLET_MAX = 256;
  private BulletImpl[] bullet = new BulletImpl[BULLET_MAX];
  private int bltIdx = 0;

  private final int ACTION_MAX = 1024;
  private ActionImpl[] action = new ActionImpl[ACTION_MAX];
  private int actIdx = 0;

  private final int FRAG_MAX = 16;
  private Frag[] frag = new Frag[FRAG_MAX];
  private int frgIdx = 0;

  private final int BATTERY_MAX = 128;
  private BatteryImpl[] battery = new BatteryImpl[BATTERY_MAX];
  private int btrIdx = 0;

  private final int OBSTACLE_MAX = 64;
  private ObstacleImpl[] obstacle = new ObstacleImpl[OBSTACLE_MAX];
  private int obstIdx = 0;

  private Ship ship;

  private Sequence4[] posSeq;
  private Sequence2[] rollSeq;

  public int cnt;

  private int posIdx, posNextCnt;
  private int scrX, scrY, scrMvX, scrMvY;
  private int rollIdx, rollNextCnt;
  private int rollD, rollMd;

  private AttractManager attractManager;

  public static final int IN_GAME = 0;
  public static final int TITLE = 1;
  public static final int GAME_OVER = 2;
  public static final int NONE = -1;

  public int status;

  // Initialize.

  public void initAll() {
    ship = new Ship(this);
    for ( int i=0 ; i<bullet.length ; i++ ) {
      bullet[i] = new BulletImpl(this, ship);
    }
    for ( int i=0 ; i<action.length ; i++ ) {
      action[i] = new ActionImpl(this);
    }
    for ( int i=0 ; i<frag.length ; i++ ) {
      frag[i] = new Frag(this);
    }
    for ( int i=0 ; i<battery.length ; i++ ) {
      battery[i] = new BatteryImpl(this);
    }
    for ( int i=0 ; i<obstacle.length ; i++ ) {
      obstacle[i] = new ObstacleImpl(this, ship);
    }
    attractManager = new AttractManager(this, ship);
  }

  private void initScroll() {
    posIdx = 0; posNextCnt = 0;
    P p = posSeq[posIdx].getP();
    scrX = -(int)(new Double(p.getX()).doubleValue()*Screen.SCREEN_WIDTH*128);
    scrY = (int)(new Double(p.getY()).doubleValue()*Screen.SCREEN_HEIGHT*128);
    scrMvX = scrX;
    scrMvY = scrY;
    if ( rollSeq != null ) {
      rollIdx = 0; rollNextCnt = 0;
      R r = rollSeq[rollIdx].getR();
      rollD = (int)(new Double(r.getD()).doubleValue()*256/360);
      rollMd = rollD;
    } else {
      rollMd = 0;
    }
  }

  public void resetAll() {
    BatteryImpl.init();
    ship.init();
    for ( int i=0 ; i<bullet.length ; i++ ) {
      bullet[i].x = BulletImpl.NOT_EXIST;
    }
    for ( int i=0 ; i<action.length ; i++ ) {
      action[i].pc = ActionImpl.NOT_EXIST;
    }
    for ( int i=0 ; i<frag.length ; i++ ) {
      frag[i].cnt = Frag.NOT_EXIST;
    }
    for ( int i=0 ; i<battery.length ; i++ ) {
      battery[i].color = BatteryImpl.NOT_EXIST;
    }
    for ( int i=0 ; i<obstacle.length ; i++ ) {
      obstacle[i].x = ObstacleImpl.NOT_EXIST;
    }
  }

  public void clearAllBullets() {
    for ( int i=0 ; i<bullet.length ; i++ ) {
      if ( bullet[i].x != BulletImpl.NOT_EXIST ) {
        bullet[i].vanishShown();
      }
    }
  }

  public void initStage() {
    initScroll();
  }

  public BulletImpl getBulletImplInstance() {
    for ( int i=BULLET_MAX-1 ; i>=0 ; i-- ) {
      bltIdx++; bltIdx &= (BULLET_MAX-1);
      if ( bullet[bltIdx].x == BulletImpl.NOT_EXIST ) {
        return bullet[bltIdx];
      }
    }
    return null;
  }

  public ActionImpl getActionImplInstance() {
    for ( int i=ACTION_MAX-1 ; i>=0 ; i-- ) {
      actIdx++; actIdx &= (ACTION_MAX-1);
      if ( action[actIdx].pc == ActionImpl.NOT_EXIST ) {
        return action[actIdx];
      }
    }
    return null;
  }


  public void addFrag(int x, int y) {
    frgIdx++; frgIdx &= (FRAG_MAX-1);
    frag[frgIdx].set(x, y);
  }

  /**
   * Load narihira stage data.
   */
  public void loadNarihiraML(String stageName, final TextArea console, String directory) {
    try {
      ErrorHandler errorHandler = new ErrorHandler() {
        public void error(SAXParseException e) {
          if ( console != null ) {
            console.append("error : ");
            console.append(e.getMessage());
            console.append("\n");
          } else {
            e.printStackTrace();
          }
        }
        public void fatalError(SAXParseException e) {
          if ( console != null ) {
            console.append("fatal error : ");
            console.append(e.getMessage());
            console.append("\n");
          } else {
            e.printStackTrace();
          }
        }
        public void warning(SAXParseException e) {
          if ( console != null ) {
            console.append("warning : ");
            console.append(e.getMessage());
            console.append("\n");
          } else {
            e.printStackTrace();
          }
        }
      };

      Document doc = null;
      try {
        /*BufferedReader rdr = new BufferedReader(
         new FileReader(".\\data\\" + stageName + File.separator + stageName + ".xml"));
        StringBuffer bml = new StringBuffer();
        for ( ; ; ) {
          String str = rdr.readLine();
          if ( str == null ) break;
          bml.append(str);
        }
        doc = jp.gr.java_conf.abagames.narihira.relaxer.UJAXP.getValidDocument(
         new StringReader(bml.toString()), errorHandler);*/
        doc = jp.gr.java_conf.abagames.narihira.relaxer.UJAXP.getValidDocument(
         new StringReader((String)NarihiraData.narihiraDatas.get(stageName)),
         errorHandler);
      } catch ( Exception e ) {
        e.printStackTrace();
      }

      jp.gr.java_conf.abagames.narihira.relaxer.Narihira nrhrML
       = new jp.gr.java_conf.abagames.narihira.relaxer.Narihira(doc);
      Scroll scr = nrhrML.getScroll();

      Sequence3[] obstSeq;
      Sequence1[] btrSeq;

      posSeq = scr.getPosition().getSequence4();
      Roll rl = scr.getRoll();
      if ( rl != null ) {
        rollSeq = rl.getSequence2();
      } else {
        rollSeq = null;
      }
      Battery btr = nrhrML.getBattery();
      if ( btr != null ) {
        btrSeq = btr.getSequence1();
        for ( btrIdx=0 ; btrIdx<btrSeq.length ; btrIdx++ ) {
          if ( btrIdx >= BATTERY_MAX ) break;
          battery[btrIdx].set(btrSeq[btrIdx].getB(), console, directory);
        }
      }
      Obstacle obst = nrhrML.getObstacle();
      if ( obst != null ) {
        obstSeq = obst.getSequence3();
        for ( obstIdx=0 ; obstIdx<obstSeq.length ; obstIdx++ ) {
          if ( obstIdx >= OBSTACLE_MAX ) break;
          obstacle[obstIdx].set(obstSeq[obstIdx].getO());
        }
      }
    } catch ( Exception e ) {
      e.printStackTrace();
    }
  }

  // Mouse handler.

  public final void setMousePos(int x, int y) {
    if ( status == IN_GAME ) {
      ship.setMousePos(x, y);
    } else {
      attractManager.setMousePos(x, y);
    }
  }

  public final void mousePressed() {
    if ( status == IN_GAME ) {
      ship.mousePressed();
    } else {
      attractManager.mousePressed();
    }
  }

  // Game lifecycle.

  private final int INTERVAL = 14;

  private Timer timerTrd;
  private Narihira nrhr;
  public Screen screen;

  public void init(Narihira nrhr, Screen screen) {
    this.nrhr = nrhr;
    this.screen = screen;
    initAll();
  }

  public void start() {
    timerTrd = new Timer(this);
    timerTrd.activate(INTERVAL, false);
  }

  public void startTitle() {
    status = TITLE;
    attractManager.initTitle();
    cnt = 0;
  }

  public void startGame() {
    status = IN_GAME;
    cnt = 0;
  }

  public void startGameover() {
    status = GAME_OVER;
    cnt = 0;
  }

  public void restart() {
    if ( timerTrd != null ) {
      timerTrd.deactivate();
    }
  }

  private synchronized void waitFrame() {
    try {
      wait();
    } catch ( InterruptedException e ) {}
  }

  public synchronized final void notifyFrame() {
    notifyAll();
  }

  private void moveBullets() {
    for ( int i=BULLET_MAX-1 ; i>=0 ; i-- ) {
      if ( bullet[i].x != BulletImpl.NOT_EXIST ) {
        bullet[i].move(scrMvX, scrMvY, rollMd);
      }
    }
  }

  private void drawBullets() {
    for ( int i=BULLET_MAX-1 ; i>=0 ; i-- ) {
      if ( bullet[i].x != BulletImpl.NOT_EXIST ) {
        bullet[i].draw();
      }
    }
  }

  private void moveFrags() {
    for ( int i=FRAG_MAX-1 ; i>=0 ; i-- ) {
      if ( frag[i].cnt != Frag.NOT_EXIST ) {
        frag[i].move();
      }
    }
  }

  private void drawFrags() {
    for ( int i=FRAG_MAX-1 ; i>=0 ; i-- ) {
      if ( frag[i].cnt != Frag.NOT_EXIST ) {
        frag[i].draw();
      }
    }
  }

  private void moveBatteries() {
    for ( int i=btrIdx-1 ; i>=0 ; i-- ) {
      if ( battery[i].color != BatteryImpl.NOT_EXIST ) {
        battery[i].move(scrMvX, scrMvY, rollMd);
      }
    }
  }

  private void moveObstacles() {
    for ( int i=obstIdx-1 ; i>=0 ; i-- ) {
      if ( obstacle[i].x != ObstacleImpl.NOT_EXIST ) {
        obstacle[i].move(scrMvX, scrMvY, rollMd);
      }
    }
  }

  private void drawObstacles() {
    for ( int i=obstIdx-1 ; i>=0 ; i-- ) {
      if ( obstacle[i].x != ObstacleImpl.NOT_EXIST ) {
        obstacle[i].draw();
      }
    }
  }

  private void moveScroll() {
    if ( cnt >= posNextCnt ) {
      posIdx++;
      if ( posIdx >= posSeq.length ) {
        startGameover();
        return;
      }
      P p = posSeq[posIdx].getP();
      int sx = -(int)(new Double(p.getX()).doubleValue()*Screen.SCREEN_WIDTH*128),
          sy = (int)(new Double(p.getY()).doubleValue()*Screen.SCREEN_HEIGHT*128),
          st = Integer.parseInt(p.getT());
      posNextCnt = st;
      scrMvX = (sx-scrX)/(st-cnt);
      scrMvY = (sy-scrY)/(st-cnt);
    }
    scrX += scrMvX; scrY += scrMvY;
    if ( rollSeq != null ) {
      if ( cnt >= rollNextCnt ) {
        rollIdx++;
        if ( rollIdx > rollSeq.length ) return;
        R r = rollSeq[rollIdx].getR();
        int sd = (int)(new Double(r.getD()).doubleValue()*(256<<8)/360),
            st = Integer.parseInt(r.getT());
        rollNextCnt = st;
        rollMd = (sd-rollD)/(st-cnt);
      }
    }
  }


  private long prvTickCount = 0;

  private void gameMove() {
    moveObstacles();
    moveBullets();
    moveBatteries();
    moveFrags();
    ship.move();
    moveScroll();
  }

  private void gameDraw() {
    drawFrags();
    drawBullets();
    drawObstacles();
    ship.draw();
  }

  public final void wakeUp() {
    if ( status == NONE ) {
      waitFrame();
      return;
    }
    screen.clear();
    long nowTick = System.currentTimeMillis();
    int frame = (int)(nowTick-prvTickCount) / INTERVAL;
    if ( frame <= 0 ) {
      frame = 1;
      prvTickCount = nowTick;
    } else if ( frame > 5 ) {
      frame = 5;
      prvTickCount = nowTick;
    } else {
      prvTickCount += frame*INTERVAL;
    }

    switch ( status ) {
    case IN_GAME:
      for ( int i=0 ; i<frame ; i++ ) {
        gameMove();
        cnt++;
      }
      gameDraw();
      break;
    case TITLE:
      for ( int i=0 ; i<frame ; i++ ) {
        //attractManager.titleMove();
        cnt++;
      }
      attractManager.titleDraw();
      break;
    case GAME_OVER:
      for ( int i=0 ; i<frame ; i++ ) {
        //attractManager.gameoverMove();
        cnt++;
      }
      attractManager.gameoverDraw();
      break;
    }
    screen.swap();

    waitFrame();
  }
}
