/* File: japanese.c */


/*
 * Copyright (c) 1997 Ben Harrison
 *
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 */

#include "angband.h"

#ifdef JP
/*
 * l strstr
 */
cptr strstr_j(const char* s, const char* t)
{
	int i, l1, l2;

	l1 = strlen(s);
	l2 = strlen(t);
	if (l1 >= l2)
	{
		for (i = 0; i <= l1 - l2; i++)
		{
			if (!strncmp(s + i, t, l2)) return s + i;
			if ( iskanji(*(s + i)) ) i++;
		}
	}
	return NULL;
}


typedef struct convert_key convert_key;

struct convert_key
{
  char *key1;
  char *key2;
};

static const convert_key s2j_table[] =
{
	{"mb","nb"},{"mp","np"},{"mv","nv"},{"mm","nm"},
	{"x","ks"},{"f$","v$"},{"ph","f"},
	/* iarwain:[C */
	{"^ia","ya-"},{"^ii","i-"},{"^iu","yu-"},{"^ie","ye-"},{"^io","yo-"},
	/* paurhach:pEAnbn */
	{"ach$","abn$"},{"uch$","ubt$"},{"och$","obz$"},
	/* sindar:V_[  parantir:peBA  feanor:tFAm[ */
	{"ar$","a-ru$"},{"ir$","ia$"},{"or$","o-ru$"},
	{"rha","ra"},{"rhi","ri"},{"rhu","ru"},{"rhe","re"},{"rho","ro"},{"rh","r"},
	{"ra",""},{"ri",""},{"ru",""},{"re",""},{"ro",""},
	{"ir","ia"},{"ur","ua"},{"er","ea"},
	{"ar","a"},
	/* quenya:NF */
	{"qua","N@"},{"qui","NB"},{"quu","ND"},{"que","NF"},{"quo","NH"},
	{"tya","`"},{"tyi","`B"},{"tyu","`"},{"tye","`F"},{"tyo","`"},
	{"sha","V"},{"shi","V"},{"shu","V"},{"she","VF"},{"sho","V"},
	{"tha","T"},{"thi","V"},{"thu","X"},{"the","Z"},{"tho","\"},{"th","X"},
	{"cha","n"},{"chi","q"},{"chu","t"},{"che","w"},{"cho","z"},{"ch","q"},
	{"dha","U"},{"dhi","W"},{"dhu","Y"},{"dhe","["},{"dho","]"},{"dh","Y"},
	{"lha",""},{"lhi",""},{"lhu",""},{"lhe",""},{"lho",""},
	{"hwa","z"},{"hwi","tC"},{"hwu","tE"},{"hwe","tG"},{"hwo","tI"},
	{"hya","q"},{"hyi","qB"},{"hyu","q"},{"hye","qF"},{"hyo","q"},
	{"ba","o"},{"bi","r"},{"bu","u"},{"be","x"},{"bo","{"},
	{"ca","J"},{"ci","L"},{"cu","N"},{"ce","P"},{"co","R"},
	{"da","_"},{"di","fB"},{"du","hD"},{"de","f"},{"do","h"},
	{"fa","t@"},{"fi","tB"},{"fu","t"},{"fe","tF"},{"fo","tH"},
	{"ga","K"},{"gi","M"},{"gu","O"},{"ge","Q"},{"go","S"},
	{"ha","n"},{"hi","q"},{"hu","t"},{"he","w"},{"ho","z"},
	{"ja","W"},{"ji","W"},{"ju","W"},{"je","WF"},{"jo","W"},
	{"ka","J"},{"ki","L"},{"ku","N"},{"ke","P"},{"ko","R"},
	{"la",""},{"li",""},{"lu",""},{"le",""},{"lo",""},
	{"ma","}"},{"mi","~"},{"mu",""},{"me",""},{"mo",""},
	{"na","i"},{"ni","j"},{"nu","k"},{"ne","l"},{"no","m"},
	{"pa","p"},{"pi","s"},{"pu","v"},{"pe","y"},{"po","|"},
	{"qu","N"},
	{"sa","T"},{"si","V"},{"su","X"},{"se","Z"},{"so","\"},
	{"ta","^"},{"ti","eB"},{"tu","gD"},{"te","e"},{"to","g"},
	{"va","@"},{"vi","B"},{"vu",""},{"ve","F"},{"vo","H"},
	{"wa",""},{"wi","EB"},{"wu","E"},{"we","EF"},{"wo","EH"},
	{"ya",""},{"yu",""},{"yo",""},
	{"za","U"},{"zi","W"},{"zu","Y"},{"ze","["},{"zo","]"},
	{"b","u"},{"c","N"},{"d","h"},{"f","t"},{"g","O"},
	{"h","t"},{"j","W"},{"k","N"},{"l",""},{"m",""},
	{"n",""},{"p","v"},{"q","N"},{"r",""},{"s","X"},
	{"t","g"},{"v",""},{"w","E"},{"y","C"},
	{"a","A"},{"i","C"},{"u","E"},{"e","G"},{"o","I"},
	{"-","["},
	{"^",""},
	{NULL,NULL}
};

const unsigned char *sindarin_to_kana(const unsigned char *sindarin)
{
	static unsigned char buf1[256], buf2[256];
	int idx;

	sprintf(buf1,"^%s$",sindarin);
	for (idx=0; buf1[idx]; idx++)
		if (isupper(buf1[idx])) buf1[idx] = tolower(buf1[idx]);

	for (idx = 0; s2j_table[idx].key1 != NULL; idx++)
	{
		unsigned char *pat1 = s2j_table[idx].key1;
		unsigned char *pat2 = s2j_table[idx].key2;
		int len = strlen(pat1);
		unsigned char *dest = buf2;
		unsigned char *src = buf1;

		while (*src)
		{
			if (strncmp(src, pat1, len) == 0)
			{
				strcpy(dest, pat2);
				src  += len;
				dest += strlen(pat2);
			}
			else if (iskanji(*src))
			{
				*dest = *src;
				dest++; src++;
				*dest = *src;
				dest++; src++;
			}
			else
			{
				*dest = *src;
				dest++; src++;
			}
		}
		*dest = 0;
		strcpy(buf1,buf2);
	}

	*strchr(buf1, '$') = '\0';

	return (buf1);
}

/*{ꓮp (łł,ł etc) */

#define CMPTAIL(y) strncmp(&in[l-strlen(y)],y,strlen(y))

/* ,R遄,R */
static void jverb1( const char *in , char *out)
{
	int l=strlen(in);
	strcpy(out,in);

	if      (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("Ă")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("ł")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("˂")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("ւ")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("ׂ")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("߂")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");

	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");

	else sprintf(&out[l],"");
}

/* ,R> ďR */
static void jverb2( const char *in , char *out)
{
	int l=strlen(in);
	strcpy(out,in);

	if      (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");

	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("Ă")==0) sprintf(&out[l-4],"Ă");
	else if (CMPTAIL("ł")==0) sprintf(&out[l-4],"ł");
	else if (CMPTAIL("˂")==0) sprintf(&out[l-4],"˂");
	else if (CMPTAIL("ւ")==0) sprintf(&out[l-4],"ւ");
	else if (CMPTAIL("ׂ")==0) sprintf(&out[l-4],"ׂ");
	else if (CMPTAIL("߂")==0) sprintf(&out[l-4],"߂");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");

	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"˂");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"ւ");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");

	else sprintf(&out[l],"Ƃɂ");
}

/* ,R > R */
static void jverb3( const char *in , char *out)
{
	int l=strlen(in);
	strcpy(out,in);

	if      (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");

	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"");
	else if (CMPTAIL("Ă")==0) sprintf(&out[l-4],"Ă");
	else if (CMPTAIL("ł")==0) sprintf(&out[l-4],"ł");
	else if (CMPTAIL("˂")==0) sprintf(&out[l-4],"˂");
	else if (CMPTAIL("ւ")==0) sprintf(&out[l-4],"ւ");
	else if (CMPTAIL("ׂ")==0) sprintf(&out[l-4],"ׂ");
	else if (CMPTAIL("߂")==0) sprintf(&out[l-4],"߂");
	else if (CMPTAIL("")==0) sprintf(&out[l-4],"ꂽ");

	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"˂");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"ւ");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");
	else if (CMPTAIL("")==0) sprintf(&out[l-2],"");

	else sprintf(&out[l],"Ƃ");
}


void jverb( const char *in , char *out , int flag)
{
	switch (flag)
	{
		case JVERB_AND:	jverb1(in, out);	break;
		case JVERB_TO:	jverb2(in, out);	break;
		case JVERB_OR:	jverb3(in, out);	break;
	}
}

static bool iskinsoku(cptr ch)
{
	cptr kinsoku[] =
	{
		"A", "B", "C", "D", "E", "H", "I",
		"R", "S", "T", "U", "X", "[", "`", 
		"j", "n", "p", "v", "x", 
		"", "", "", "", "", "", "", "", "",
		"@", "B", "D", "F", "H", "b", "", "", "",
		NULL
	};
	int i;

	for (i = 0; kinsoku[i]; i++)
	{
		if (streq(ch, kinsoku[i]))
		{
			return TRUE;
		}
	}
	return FALSE;
}

/* Find a reasonable break-point */
int find_break_j(cptr str, const int witdh, const bool wrap)
{
	int i = 0;
	int bpos = -1;
	char ch[3];

	ch[2] = '\0';

	while (str[i] != 0 && i < ((iskanji(str[i])) ? witdh - 2 : witdh - 1))
	{
		bool k_flg = iskanji(str[i]);

		i += k_flg ? 2 : 1;

		if (wrap)
		{
			if (str[i] == '\0') bpos = i;

			else if (iskanji(str[i]))
			{
				ch[0] = str[i];
				ch[1] = str[i+1];
				if (!iskinsoku(ch)) bpos = i;
			}

			else if (k_flg || strchr(" \n(", str[i]))
			{
				bpos = i;
			}

			if (str[i] == '\n') break;
		}

	}

	if (bpos == -1) bpos = i;

	return bpos;
}

void roff_to_buf(cptr str, int width, char *buf)
{
	int n = -1;
	cptr s;
	char *d;

	s = str;
	d = buf;

	while (*s)
	{
		/* Skip white spaces */
		while (!isgraph(*s) && !iskanji(*s)) s++;

		/* Find a reasonable break-point */
		n = find_break_j(s, width, TRUE);

		/* Copy one line of history */
		strncpy(d, s, n);
		d[n] = '\0';
		d += n + 1;
		s += n;
	}

	/* Add a blank line */
	d[0] = '\0';
}

cptr jnumsuffix(object_type *o_ptr)
{
	switch (o_ptr->tval)
	{
		case TV_BOLT:
		case TV_ARROW:
		case TV_POLEARM:
		case TV_STAFF:
		case TV_WAND:
		case TV_ROD:
		case TV_DIGGING:
		{
			return  "{";
		}
		case TV_SCROLL:
		{
			return  "";
		}
		case TV_POTION:
		{
			return  "";
		}

		case TV_SORCERY_BOOK:
		case TV_NATURE_BOOK:
		case TV_CHAOS_BOOK:
		case TV_DEATH_BOOK:
		case TV_TRUMP_BOOK:
		case TV_ARCANE_BOOK:
		{
			return  "";
		}
		case TV_SOFT_ARMOR:
		case TV_HARD_ARMOR:
		case TV_DRAG_ARMOR:
		case TV_CLOAK:
		{
			return  "";
		}
		case TV_SWORD:
		case TV_HAFTED:
		case TV_BOW:
		{
			return  "U";
		}
		case TV_BOOTS:
		{
			return  "";
		}
		/* Hׂ by ita */
		case TV_FOOD:
		{
			if (o_ptr->sval == SV_FOOD_JERKY)
			{
				return "؂";
			}
			break;
		}
		/* O[u by 퐣 */
		case TV_GLOVES:
		{
			return  "g";
		}
	}

	if (o_ptr->number < 10)
	{
		return  "";
	}
	else
	{
		return  "";
	}
}

void codeconv(char *str)
{
	int i;
	int kanji, iseuc;
	unsigned char c1, c2;
	unsigned char tmp[256];

	/* ݂ÅR[hEUCǂׂB*/
	kanji = 0;
	iseuc = 1;
	for (i = 0; str[i]; i++)
	{
		c1 = str[i];
		if (c1 & 0x80)  kanji = 1;
		if ( c1>=0x80 && (c1 < 0xa1 || c1 > 0xfe)) iseuc = 0;
	}

#ifdef EUC
	if (kanji && !iseuc)	 /* SJIS -> EUC */
	{
		for (i = 0; str[i]; i++)
		{
			c1 = str[i];
			if (c1 & 0x80)
			{
				i++;
				c2 = str[i];
				if (c2 >= 0x9f)
				{
					c1 = c1 * 2 - (c1 >= 0xe0 ? 0xe0 : 0x60);
					c2 += 2;
				}
				else
				{
					c1 = c1 * 2 - (c1 >= 0xe0 ? 0xe1 : 0x61);
					c2 += 0x60 + (c2 < 0x7f);
				}
				tmp[i - 1] = c1;
				tmp[i] = c2;
			}
			else
				tmp[i] = c1;
		}
		tmp[i] = '\0';
		strcpy(str, tmp);
	}
#endif

#ifdef SJIS
	if (kanji && iseuc)	/* EUC -> SJIS */
	{
		for (i = 0; str[i]; i++)
		{
			c1 = str[i];
			if (c1 & 0x80)
			{
				i++;
				c2 = str[i];
				if (c1 % 2)
				{
					c1 = (c1 >> 1) + (c1 < 0xdf ? 0x31 : 0x71);
					c2 -= 0x60 + (c2 < 0xe0);
				}
				else
				{
					c1 = (c1 >> 1) + (c1 < 0xdf ? 0x30 : 0x70);
					c2 -= 2;
				}
				tmp[i - 1] = c1;
				tmp[i] = c2;
			}
			else
				tmp[i] = c1;
		}
		tmp[i] = '\0';
		strcpy(str, tmp);
	}
#endif
}

/*
 * "of Foo""Foo",A
 * "'Bar'""wBarx"ɕϊB
 */
const unsigned char *e_to_j(const unsigned char *str)
{
	static char buf[256];
	int len = strlen(str);

	if (strncmp(str, "of ", 3) == 0)
	{
		strcpy(buf, &str[3]);
		strcat(buf, "");
	}
	else if (str[0] == '\'' && str[len-1] == '\'')
	{
		strcpy(buf, "w");
		strncat(buf, &str[1], len-2);
		strcat(buf, "x");
	}
	else
	{
		strcpy(buf, str);
	}

	return buf;
}

#endif
