/*
 * $Id: PreviewerCanvas.java,v 1.2 2003/01/24 14:43:51 kenta Exp $
 *
 * Copyright 2003 Kenta Cho. All rights reserved.
 */
package jp.gr.java_conf.abagames.bulletnote;

import java.io.IOException;
import java.io.StringReader;
import java.util.Vector;

import javax.xml.parsers.ParserConfigurationException;
import jp.gr.java_conf.abagames.bulletml.Action;
import jp.gr.java_conf.abagames.bulletml.Bullet;
import jp.gr.java_conf.abagames.bulletml.Bulletml;
import jp.gr.java_conf.abagames.bulletml.BulletmlEntityResolver;
import jp.gr.java_conf.abagames.bulletml.BulletmlManager;
import jp.gr.java_conf.abagames.bulletml.Fire;
import jp.gr.java_conf.abagames.bulletml.IActionElmChoice;
import jp.gr.java_conf.abagames.bulletml.IBulletmlChoice;
import jp.gr.java_conf.abagames.bulletml.IBulletmlScreen;
import jp.gr.java_conf.abagames.bulletml.UJAXP;
import jp.gr.java_conf.abagames.util.ITime;
import jp.gr.java_conf.abagames.util.Timer;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * BulletML previewer screen handler and timer .
 *
 * @version $Revision: 1.2 $
 */
public class PreviewerCanvas extends Canvas implements IBulletmlScreen, ITime {//Runnable {
	private BulletmlManager manager;

	private Display display;
  private GC gc;
	private Color background;
	private final int BULLET_COLOR_NUM = 3;
	private Color[] bulletColor = new Color[BULLET_COLOR_NUM];
  private Point size;
  
  public PreviewerCanvas(Composite parent, int style) {
    super(parent, style);
    
    addControlListener(new ControlAdapter() {
      public void controlResized(ControlEvent e) {
        resized();
      }
    });
    addDisposeListener(new DisposeListener() {
      public void widgetDisposed(DisposeEvent e){
        stop();
      }
    });
    display = getDisplay();
    gc = new GC(this);
		bulletColor[0] = new Color(display, 255, 200, 200);
		bulletColor[1] = new Color(display, 200, 255, 200);
		bulletColor[2] = new Color(display, 200, 200, 255);
		background = new Color(display, 0, 0, 0);
		gc.setBackground(background);

		manager = new BulletmlManager();
		manager.init();
  }

  private void resized() {
    size = getSize();
		manager.setScreen(this);
		int sx, sy;
		if ( !horizontal ) {
			sx = size.x<<7; sy = (size.y<<6)*3;
		} else {
			sx = size.x<<6; sy = size.y<<7;
		}
		manager.setShipPos(sx, sy, sx, sy+1);
  }

  public void stop() {
    deactive();
    for ( int i=0 ; i<BULLET_COLOR_NUM ; i++ ) {
	    bulletColor[i].dispose();
    }
    background.dispose();
    gc.dispose();
  }

	public void setRank(float rank) {
		manager.setRank(rank);
	}
	  
	private final String TOP_PREFIX = "top";
	private ErrorHandler errorHandler;
	private boolean horizontal;

	public void setBulletML(String document, final IResource resource) 
	 throws SAXException, ParserConfigurationException, IOException {
    errorHandler = new ErrorHandler() {
      public void error(SAXParseException e) {
      	try {
					IMarker marker = resource.createMarker("org.eclipse.core.resources.problemmarker");
					marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
					marker.setAttribute(IMarker.MESSAGE, e.getMessage());
					marker.setAttribute(IMarker.LINE_NUMBER, e.getLineNumber());
      	} catch ( CoreException ex ) {
      		ex.printStackTrace();
      	}
      }
      public void fatalError(SAXParseException e) {
      	try {
					IMarker marker = resource.createMarker("org.eclipse.core.resources.problemmarker");
					marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
					marker.setAttribute(IMarker.MESSAGE, e.getMessage());
					marker.setAttribute(IMarker.LINE_NUMBER, e.getLineNumber());
      	} catch ( CoreException ex ) {
      		ex.printStackTrace();
      	}
      }
      public void warning(SAXParseException e) {
      	try {
					IMarker marker = resource.createMarker("org.eclipse.core.resources.problemmarker");
					marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO);
					marker.setAttribute(IMarker.MESSAGE, e.getMessage());
					marker.setAttribute(IMarker.LINE_NUMBER, e.getLineNumber());
      	} catch ( CoreException ex ) {
      		ex.printStackTrace();
      	}
      }
    };
    clearScreen();
		deactive();
		manager.reinit();
		manager.setErrorHandler(errorHandler);

    Document doc = UJAXP.getDocument(
     new StringReader(document), UJAXP.FLAG_VALIDATION, errorHandler, new BulletmlEntityResolver());
		Bulletml bulletml = new Bulletml(doc);
		if ( bulletml.getType().equals("horizontal") ) {
			horizontal = true;
		} else {
			horizontal = false;
		}
		Vector actionsVct = new Vector();
		IBulletmlChoice[] choices = bulletml.getContent();
		for ( int i=0 ; i<choices.length ; i++ ) {
			if ( choices[i] instanceof Action ) {
				Action action = (Action)choices[i];
				String label = action.getLabel();
				if ( label != null && label.startsWith(TOP_PREFIX) ) {
					actionsVct.addElement(action);
				} else {
					manager.bulletmlUtil.addAction(action);
				}
			} else if ( choices[i] instanceof Bullet ) {
				manager.bulletmlUtil.addBullet((Bullet)choices[i]);
			} else if ( choices[i] instanceof Fire ) {
				manager.bulletmlUtil.addFire((Fire)choices[i]);
			}
		}
		IActionElmChoice[] actions = new IActionElmChoice[actionsVct.size()];
		actionsVct.copyInto(actions);
		manager.setTopActions(actions);
		
		resized();
		restartCount = 0;
		start();
	}

	public int getWidth() {
    return size.x;
	}
	  
	public int getHeight() {
    return size.y;
	}

	public void hitShip() {
	}
	
	public void drawBullet(int x, int y, int px, int py, int sx, int sy, int cl, int cnt) {
		gc.setForeground(bulletColor[cl%BULLET_COLOR_NUM]);
		gc.drawLine(x>>8, y>>8, px>>8, py>>8);
	}
		  
  /*
   * Methods to handle a thread.
   */
  private Timer timer;
  private int interval = 33;

  public void start() {
  	deactive();
  	timer = new Timer(this);
    timer.activate(interval, false);
  }

  public void setInterval(int itv) {
    interval = itv;
    timer.setDuration(itv);
  }

  public void deactive() {
		if ( timer != null ) {
	    timer.deactivate();
		}
  }

  public void terminate() {
    timer.terminate();
  }

  private synchronized void waitFrame() {
    try {
      wait();
    } catch ( InterruptedException e ) {}
  }

  public synchronized final void notifyFrame() {
    notifyAll();
  }

  public final void wakeUp() {
    try {
      loop();
    } catch ( Exception e ) {
    	try {
	    	errorHandler.fatalError(new SAXParseException("runtime error: " + e.toString(), null));
    	} catch ( SAXException ex ) {}
    	deactive();
    }
    waitFrame();
  }

	private void clearScreen() {
		if ( gc == null || size == null ) return;
    gc.fillRectangle(0, 0, size.x, size.y);
	}
	
	private final int DEFAULT_DEGREE = 180;

  private long prvTickCount = 0;
	private int restartCount;

  /**
   * Thread main loop.
   */
  private void loop() {
    long nowTick = System.currentTimeMillis();
    int frame = (int)(nowTick-prvTickCount) / interval;
    if ( frame <= 0 ) {
      frame = 1;
      prvTickCount = nowTick;
    } else if ( frame > 5 ) {
      frame = 5;
      prvTickCount = nowTick;
    } else {
      prvTickCount += frame*interval;
    }
    for ( int i=0 ; i<frame ; i++ ) {
			manager.moveBullets();
    }
    if ( manager.isFinished() ) {
    	restartCount--;
    	if ( restartCount < 0 ) {
    		restartCount = 60;
    		if ( !horizontal ) {
					manager.addTopBullet(size.x<<7, size.y<<6, DEFAULT_DEGREE);
    		} else {
					manager.addTopBullet((size.x<<6)*3, size.y<<7, DEFAULT_DEGREE);
    		}
    	}
    }
		clearScreen();
    manager.drawBullets();
	}
}
