/*
 * $Id: BulletmlUtil.java,v 1.1.1.1 2003/01/24 14:34:00 kenta Exp $
 *
 * Copyright 2002 Kenta Cho. All rights reserved.
 */
package jp.gr.java_conf.abagames.bulletml;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * Utility class for BulletML.
 *
 * @version $Revision: 1.1.1.1 $
 */
public class BulletmlUtil {
	private ErrorHandler errorHandler;
	
	public void setErrorHandler(ErrorHandler handler) {
		errorHandler = handler;
	}
	
  private Hashtable expressions = new Hashtable();

  public int getIntValue(String v, float[] prms) {
    return (int)evalExpression(v, prms);
  }

  public float getFloatValue(String v, float[] prms) {
    return evalExpression(v, prms);
  }

  private char[] expChr;
  private float rank = 0.5f;

  public void setRank(float rk) {
    rank = rk;
  }

  public float getRank() {
    return rank;
  }

  private void evalFloatValue(Expression ep, int stIdx, int lgt, float sign) {
    if ( expChr[stIdx] == '$' ) {
      String label = new String(expChr, stIdx+1, lgt-1);
      if ( label.equals("rand") ) {
        ep.push(0, Expression.STACK_RAND);
      } else if ( label.equals("rank") ) {
        ep.push(0, Expression.STACK_RANK);
      } else {
        int idx;
        try {
          idx = new Integer(label).intValue()-1;
        } catch ( NumberFormatException e ) {
          ep.push(0, Expression.STACK_NUM);
          return;
        }
        ep.push(0, Expression.STACK_VARIABLE+idx);
      }
    } else {
      try {
        ep.push(new Float(new String(expChr, stIdx, lgt)).floatValue()*sign,
         Expression.STACK_NUM);
      } catch ( NumberFormatException e ) {
        ep.push(0, Expression.STACK_NUM);
      }
    }
  }

  private void evalExpPart(Expression ep, int stIdx, int edIdx) {
    int op[] = new int[] {-1, -1};
    while (expChr[stIdx]=='(' && expChr[edIdx-1]==')') {
      stIdx++;
      edIdx--;
    }
    for ( int i = edIdx-1; i >= stIdx ; i-- ) {
      char c = expChr[i];
      if ( c == ')' ) {
        do {
          i--;
        } while ( expChr[i] != '(' );
      } else if ( op[0] < 0 && (c=='*' || c=='/' || c=='%') ) {
        op[0] = i;
      } else if ( c=='+' || c=='-' ) {
        op[1] = i;
        break;
      }
    }
    if ( op[1] < 0 ) {
      if ( op[0] < 0 ) {
        evalFloatValue(ep, stIdx, edIdx-stIdx, 1);
      } else {
        switch(expChr[op[0]]) {
        case '*':
          evalExpPart(ep, stIdx, op[0]);
          evalExpPart(ep, op[0]+1, edIdx);
          ep.setOperator(Expression.MULTIPLE);
          break;
        case '/':
          evalExpPart(ep, stIdx, op[0]);
          evalExpPart(ep, op[0]+1, edIdx);
          ep.setOperator(Expression.DIVISION);
          break;
        case '%':
          evalExpPart(ep, stIdx, op[0]);
          evalExpPart(ep, op[0]+1, edIdx);
          ep.setOperator(Expression.MODULO);
          break;
        }
      }
    } else {
      if ( op[1] == stIdx ) {
        switch (expChr[op[1]]) {
        case '-':
          evalFloatValue(ep, stIdx+1, edIdx-stIdx-1, -1);
          break;
        case '+':
          evalFloatValue(ep, stIdx+1, edIdx-stIdx-1, 1);
          break;
        }
      } else {
        switch(expChr[op[1]]) {
        case '+':
          evalExpPart(ep, stIdx, op[1]);
          evalExpPart(ep, op[1]+1, edIdx);
          ep.setOperator(Expression.PLUS);
          break;
        case '-':
          evalExpPart(ep, stIdx, op[1]);
          evalExpPart(ep, op[1]+1, edIdx);
          ep.setOperator(Expression.MINUS);
          break;
        }
      }
    }
  }

  public float evalExpression(String exp, float[] p) {
    Expression ep = (Expression)expressions.get(exp);
    if ( ep == null ) {
      expChr = new char[exp.length()];
      int ecIdx = 0;
      boolean skip = false;
      StringBuffer buf = new StringBuffer(exp);
      int depth = 0;
      boolean balance = true;
      char ch;
      for( int i=0 ; i<buf.length() ; i++ ) {
        ch = buf.charAt(i);
        switch(ch) {
        case ' ':
        case '\n':
          skip = true;
          break;
        case ')':
          depth--;
          if ( depth < 0 ) balance = false;
          break;
        case '(':
          depth++;
          break;
        }
        if ( skip ) {
          skip = false;
        } else {
          expChr[ecIdx] = ch;
          ecIdx++;
        }
      }
      if ( depth != 0 || !balance ) {
        return 0;
      }
      ep = new Expression(this);
      evalExpPart(ep, 0, ecIdx);
      expressions.put(exp, ep);
    }
    return ep.calc(p);
  }

  private Hashtable bullets = new Hashtable();
  private Hashtable actions = new Hashtable();
  private Hashtable fires = new Hashtable();

  public void clear() {
    bullets.clear();
    actions.clear();
    fires.clear();
    expressions.clear();
  }

  public void addBullet(Bullet blt) {
    bullets.put(blt.getLabel(), blt);
  }

  public void addAction(Action act) {
    actions.put(act.getLabel(), act);
  }

  public void addFire(Fire fre) {
    fires.put(fre.getLabel(), fre);
  }

  public final Bullet getBulletElm(IBulletElmChoice bec) {
    if ( bec instanceof BulletRef ) {
      String label = ((BulletRef)bec).getLabel();
      Bullet blt = (Bullet)bullets.get(label);
      if ( blt == null ) {
      	if ( errorHandler == null ) {
	        System.out.println("unknown bullet label: " + label);
      	} else {
      		try {
	      		errorHandler.fatalError(new SAXParseException("unknown bullet label: " + label, null));
      		} catch ( SAXException e ) {}
      	}
      }
      return blt;
    } else if ( bec instanceof Bullet ) {
      return (Bullet)bec;
    }
    return null;
  }

  public final float[] getBulletParams(IBulletElmChoice bec, float[] prms) {
    if ( bec instanceof BulletRef ) {
      BulletRef br = (BulletRef)bec;
      float[] prm = new float[br.getParamCount()];
      for ( int i=prm.length-1; i>=0 ; i-- ) {
        prm[i] = getFloatValue(br.getParam(i).getContent(), prms);
        //prm[i] = getFloatValue(br.getParam(i), prms);
      }
      return prm;
    }
    return null;
  }

  public final Action getActionElm(IActionElmChoice aec) {
    if ( aec instanceof ActionRef ) {
      String label = ((ActionRef)aec).getLabel();
      Action act = (Action)actions.get(label);
      if ( act == null ) {
      	if ( errorHandler == null ) {
	        System.out.println("unknown action label: " + label);
      	} else {
      		try {
	      		errorHandler.fatalError(new SAXParseException("unknown action label: " + label, null));
      		} catch ( SAXException e ) {}
      	}
      }
      return act;
    } else if ( aec instanceof Action ) {
      return (Action)aec;
    }
    return null;
  }

  public final float[] getActionParams(IActionElmChoice aec, float[] prms) {
    if ( aec instanceof ActionRef ) {
      ActionRef ar = (ActionRef)aec;
      float[] prm = new float[ar.getParamCount()];
      for ( int i=prm.length-1; i>=0 ; i-- ) {
        //prm[i] = getFloatValue(ar.getParam(i).getContent(), prms);
        prm[i] = getFloatValue(ar.getParam(i), prms);
      }
      return prm;
    }
    return null;
  }

  public final Fire getFireElm(IActionChoice fec) {
    if ( fec instanceof FireRef ) {
      String label = ((FireRef)fec).getLabel();
      Fire fire = (Fire)fires.get(label);
      if ( fire == null ) {
      	if ( errorHandler == null ) {
	        System.out.println("unknown fire label: " + label);
      	} else {
      		try {
	      		errorHandler.fatalError(new SAXParseException("unknown fire label: " + label, null));
      		} catch ( SAXException e ) {}
      	}
      }
      return fire;
    } else if ( fec instanceof Fire ) {
      return (Fire)fec;
    }
    return null;
  }

  public final float[] getFireParams(IActionChoice fec, float[] prms) {
    if ( fec instanceof FireRef ) {
      FireRef fr = (FireRef)fec;
      float[] prm = new float[fr.getParamCount()];
      for ( int i=prm.length-1; i>=0 ; i-- ) {
        prm[i] = getFloatValue(fr.getParam(i).getContent(), prms);
        //prm[i] = getFloatValue(fr.getParam(i), prms);
      }
      return prm;
    }
    return null;
  }

  public final IActionElmChoice[] getActionStartsWith(String label) {
    Vector actVct = new Vector();
    Enumeration keys = actions.keys();
    while ( keys.hasMoreElements() ) {
      String lbl = (String)keys.nextElement();
      if ( lbl.startsWith(label) ) {
        actVct.addElement((Action)actions.get(lbl));
      }
    }
    IActionElmChoice[] iac = new IActionElmChoice[actVct.size()];
    actVct.copyInto(iac);
    return iac;
  }
}
